"use client";

import {zodResolver} from "@hookform/resolvers/zod";
import {useForm} from "react-hook-form";
import {z} from "zod";

import {PhoneInput} from "react-international-phone";
import "react-international-phone/style.css";

import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";

import {Button} from "@/components/ui/button";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {Input} from "@/components/ui/input";
import {Textarea} from "../ui/textarea";
import {useState, useTransition} from "react";
import {toast} from "sonner";
import {AiOutlineLoading3Quarters} from "react-icons/ai";
import Image from "next/image";
import {bookingFormSchema} from "@/schemas";
import {postPayment} from "@/app/api/postActions";
import {sendGAEvent} from "@next/third-parties/google";
import {X} from "lucide-react";

interface Props {
  data: any;
}
const FormGuest = ({data}: Props) => {
  const [isPending, startTransition] = useTransition();
  const [loading, setLoading] = useState(false);

  const form = useForm<z.infer<typeof bookingFormSchema>>({
    resolver: zodResolver(bookingFormSchema),
    defaultValues: {
      start: data.start,
      end: data.end,
      adult: data.adult,
      child: data.child,
      usd: data.usd,
      idr: data.idr,
      firstname: "",
      lastname: "",
      phone: "",
      email: "",
      address: "No Address",
      message: "",
    },
  });

  const onSubmit = (values: z.infer<typeof bookingFormSchema>) => {
    setLoading(true);
    startTransition(async () => {
      const response = await postPayment(values);
      if (response.success) {
        toast.success("Redirecting to checkout page...");
        window.location.href = response.data;
      } else {
        toast.custom((t: any) => (
          <div className="flex justify-between p-4 min-w-[370px] gap-x-5 bg-white">
            <div className="text-start space-y-1">
              <p className="text-sm text-main">{response.message}</p>
            </div>
            <button
              type="button"
              onClick={() => toast.dismiss(t)}
              className="bg-main p-1 rounded h-fit"
            >
              <X size={20} className="text-second" />
            </button>
          </div>
        ));
        setTimeout(() => {
          window.location.reload();
        }, 3000);
      }
    });
  };
  return (
    <>
      <div className="col-span-1 lg:col-span-5 h-full relative">
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)}>
            <Card className="rounded">
              {loading && (
                <div className="absolute top-2 left-2 w-[98%] h-[97%] z-50">
                  <div className="flex flex-col items-center justify-center h-full w-full bg-white/80 gap-3">
                    <AiOutlineLoading3Quarters className="h-10 w-10 animate-spin" />
                    <span className="text-xl font-bold">
                      Waiting for payment
                    </span>
                    <p className="text-base md:text-lg ">
                      Complete your payment on the next checkout page to confirm
                      your booking.
                    </p>
                  </div>
                </div>
              )}
              <CardHeader>
                <CardTitle className="text-second text-2xl">
                  Guest Detail
                </CardTitle>
                <CardDescription className="text-main">
                  Fill your details here
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="grid lg:grid-cols-2 gap-3">
                  <FormField
                    control={form.control}
                    name="firstname"
                    render={({field}) => (
                      <FormItem>
                        <FormLabel className="text-main after:content-['*'] after:text-red-500">
                          {" "}
                          First Name
                        </FormLabel>
                        <FormControl>
                          <Input
                            placeholder="Enter your first name"
                            {...field}
                            autoComplete="current-first name"
                            type="text"
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="lastname"
                    render={({field}) => (
                      <FormItem>
                        <FormLabel className="text-main after:content-['*'] after:text-red-500">
                          {" "}
                          Last Name
                        </FormLabel>
                        <FormControl>
                          <Input
                            placeholder="Enter your last name"
                            {...field}
                            autoComplete="current-last name"
                            type="text"
                            {...field}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="email"
                    render={({field}) => (
                      <FormItem>
                        <FormLabel className="text-main after:content-['*'] after:text-red-500">
                          {" "}
                          Email
                        </FormLabel>
                        <FormControl>
                          <Input
                            placeholder="Enter your email"
                            {...field}
                            autoComplete="current-email"
                            type="email"
                            {...field}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="phone"
                    render={({field: {onChange, value}}) => (
                      <FormItem>
                        <FormLabel className="text-main after:content-['*'] after:text-red-500">
                          {" "}
                          Phone or Whatsapp
                        </FormLabel>
                        <FormControl>
                          <PhoneInput
                            placeholder="Enter your phone number"
                            className="w-full !h-[40px] !rounded-md"
                            defaultCountry="id"
                            value={value}
                            onChange={onChange}
                            countrySelectorStyleProps={{
                              buttonClassName:
                                "!h-full !border-[#e2e8f0] !px-3 z-0",
                              dropdownStyleProps: {
                                className: "!border-none ",
                              },
                            }}
                            inputClassName="w-full !h-full !border-[#e2e8f0]"
                            forceDialCode={true}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <div className="lg:col-span-2">
                    <FormField
                      control={form.control}
                      name="message"
                      render={({field}) => (
                        <FormItem>
                          <FormLabel className="text-main">Notes</FormLabel>
                          <FormControl>
                            <Textarea
                              placeholder="Enter additional details for your booking"
                              className="resize-none"
                              {...field}
                            />
                          </FormControl>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  </div>
                </div>
              </CardContent>
              <CardFooter className="flex flex-col gap-2">
                <div className="flex items-center justify-between w-full">
                  <small className="before:content-['*'] before:text-red-500">
                    {" "}
                    - Required
                  </small>
                  <Button
                    type="submit"
                    className="bg-main hover:bg-main"
                    onClick={async () => {
                      sendGAEvent("event", "begin_checkout", {
                        value: data?.usd,
                        currency: "USD",
                        items: [
                          {
                            item_name: "Villa La Mainson",
                            price: data?.price[0].rate,
                            quantity: data?.nights,
                          },
                        ],
                      });
                    }}
                  >
                    Confirm Booking
                  </Button>
                </div>
                <div className="flex flex-col gap-2">
                  <small className="before:content-['*'] before:text-red-500 text-justify">
                    {" "}
                    You're going to be redirected to our trusted payment
                    partner, <span className="font-extrabold">Xendit</span>, to
                    complete your transaction.{" "}
                    <span className="font-extrabold">
                      For a smooth booking experience, please follow the steps
                      until you're safely redirected back to our page.
                    </span>{" "}
                    Rest assured, your payment is protected. We’ll finalize your
                    booking as soon as the process is complete!
                  </small>
                </div>
              </CardFooter>
            </Card>
          </form>
        </Form>
      </div>
      <div className="col-span-1 lg:col-span-2 h-full">
        <Card className="rounded">
          <CardHeader>
            <CardTitle className="text-second text-2xl">Payment Method</CardTitle>
            <CardDescription className="text-main text-justify">
              All payments on this website are denominated in {""}
              <span className="font-bold">
                Indonesian Rupiah (IDR). Prices in USD only served to help you
                to estimate the payment you will make.
              </span>
            </CardDescription>
          </CardHeader>
          <CardContent>
            <h3 className="font-semibold leading-none tracking-tight text-second text-2xl mb-2">
              Description
            </h3>
            <ul className="max-w-md space-y text-gray-500 list-none list-inside">
              <li>
                <div className="flex justify-between items-center">
                  <p className="text-sm">Credit Card</p>
                  <div className="flex items-center justify-between gap-3">
                    <Image
                      src="/checkout/svg/visa.svg"
                      width={30}
                      height={30}
                      alt="visa"
                      loading="lazy"
                    />
                    <Image
                      src="/checkout/svg/mastercard.svg"
                      width={30}
                      height={30}
                      alt="mastercard"
                      loading="lazy"
                    />
                    <Image
                      src="/checkout/svg/jcb.svg"
                      width={30}
                      height={30}
                      alt="jcb"
                      loading="lazy"
                    />
                  </div>
                </div>
              </li>
              <li>
                <div className="flex justify-between items-center">
                  <p className="text-sm">BNI Bank</p>
                  <Image
                    src="/checkout/svg/bni.svg"
                    width={30}
                    height={30}
                    alt="bni"
                    loading="lazy"
                  />
                </div>
              </li>
              <li>
                <div className="flex justify-between items-center">
                  <p className="text-sm">BRI Bank</p>
                  <Image
                    src="/checkout/svg/bri.svg"
                    width={30}
                    height={30}
                    alt="bri"
                    loading="lazy"
                  />
                </div>
              </li>
              <li>
                <div className="flex justify-between items-center">
                  <p className="text-sm">Mandiri Bank</p>
                  <Image
                    src="/checkout/svg/mandiri.svg"
                    width={30}
                    height={30}
                    alt="mandiri"
                    loading="lazy"
                  />
                </div>
              </li>
              <li>
                <div className="flex justify-between items-center">
                  <p className="text-sm">Permata Bank</p>
                  <Image
                    src="/checkout/svg/permata.svg"
                    width={30}
                    height={30}
                    alt="permata"
                    loading="lazy"
                  />
                </div>
              </li>
            </ul>
          </CardContent>
        </Card>
      </div>
    </>
  );
};

export default FormGuest;
