"use client";

import {formSchema} from "@/schemas";
import {format, toZonedTime} from "date-fns-tz";
import {useState, useTransition} from "react";
import {Range} from "react-date-range";
import {useForm} from "react-hook-form";
import {z} from "zod";
import {zodResolver} from "@hookform/resolvers/zod";
// import {getDisabledDates} from "@/lib/disableDates";
import {isSameDay} from "date-fns";
import {toast} from "sonner";
import {X} from "lucide-react";
import {Form, FormControl, FormField, FormItem, FormMessage} from "../ui/form";
import {Card, CardContent, CardFooter, CardHeader, CardTitle} from "../ui/card";
import Link from "next/link";
import {
  IoArrowBackCircleOutline,
  IoCalendarOutline,
  IoLockClosed,
} from "react-icons/io5";
import {Input} from "../ui/input";
import {Popover, PopoverContent, PopoverTrigger} from "../ui/popover";
import {PopoverClose} from "@radix-ui/react-popover";
import CalendarWidget from "../calendar-widget";
import Counter from "../counter";
import {Button} from "../ui/button";
import {FaRegUser} from "react-icons/fa";
import ResultForm from "./ResultForm";
import {postAvailability} from "@/app/api/postActions";
import FormGuest from "./FormGuest";
import { useGet } from "@/hooks/use-Query";
import { getIcal } from "@/app/api/getActions";

const date = new Date();

const timeZone = "Asia/Jakarta";
const zonedDate = toZonedTime(date, timeZone);

export default function FormAvailability() {
  const [isPending, startTransition] = useTransition();

    const {data: icalVilla, isLoading} = useGet({
    key: `ical`,
    fn: () => getIcal(),
  });

  const [openCalendar, setOpenCalendar] = useState(false);
  const [openGuestForm, setOpenGuestForm] = useState(false);
  const [adult, setAdult] = useState(1);
  const [child, setChild] = useState(0);

  const [arrivalDate, setArrivalDate] = useState<Date>(new Date());
  const [departureDate, setDepartureDate] = useState<Date>(new Date());

  const [formData, setFormData] = useState<any>(null);
  const [result, setResult] = useState<any>(null);

  const [dateRange, setDateRange] = useState<Range>({
    startDate: zonedDate,
    endDate: zonedDate,
    key: "selection",
  });

  const [guest, setGuest] = useState<any>({
    adult: 1,
    child: 0,
  });

  const totalGuests = guest.adult + guest.child;

  const form = useForm<z.infer<typeof formSchema>>({
    resolver: zodResolver(formSchema),
  });

  const handleDateChange = (value: any) => {
    setDateRange({
      startDate: value.selection.startDate,
      endDate: value.selection.endDate,
      key: "selection",
    });

    if (
      value.selection.startDate.toDateString() !==
      value.selection.endDate.toDateString()
    ) {
      setOpenCalendar(false);
      form.setValue("start", format(value.selection.startDate, "yyyy-MM-dd"));
      form.setValue("end", format(value.selection.endDate, "yyyy-MM-dd"));
    }
  };

  const handleCounter = (value: number, name: string) => {
    if (name === "adult") {
      setGuest({
        adult: value,
        child: guest.child,
      });
      form.setValue("adult", Number(value));
    }

    if (name === "child") {
      setGuest({
        adult: guest.adult,
        child: value,
      });
      form.setValue("child", Number(value));
    }
  };

  const onSubmit = (values: z.infer<typeof formSchema>) => {
    if (isSameDay(dateRange.startDate as Date, dateRange.endDate as Date)) {
      setOpenCalendar(true);
      return;
    } else {
         startTransition(async () => {
        // toast(<pre>{JSON.stringify(values, null, 2)}</pre>);
        const response = await postAvailability(values);

        if (response.success) {
          setResult(response);
          const resData = response.data;
          const data = {
            ...values,
            ...resData,
          };
          setFormData(data);
          setArrivalDate(dateRange.startDate as Date);
          setDepartureDate(dateRange.endDate as Date);
          setAdult(guest.adult);
          setChild(guest.child);
          form.setValue(
            "start",
            format(dateRange.startDate as Date, "yyyy-MM-dd")
          );
          form.setValue("end", format(dateRange.endDate as Date, "yyyy-MM-dd"));
          form.setValue("adult", Number(guest.adult));
          form.setValue("child", Number(guest.child));
        } else {
          toast.custom((t: any) => (
            <div className="flex justify-between p-4 min-w-[370px] gap-x-5 bg-second">
              <div className="text-start space-y-1">
                <p className="text-xs text-main"></p>
                <p className="text-sm text-main">{response.message}</p>
              </div>
              <button
                type="button"
                onClick={() => toast.dismiss(t)}
                className="bg-main p-1 rounded h-fit"
              >
                <X size={20} className="text-second" />
              </button>
            </div>
          ));
          setDateRange({
            startDate: arrivalDate,
            endDate: departureDate,
            key: "selection",
          });
          setGuest({
            adult: adult,
            child: child,
          });

          form.setValue("start", format(arrivalDate as Date, "yyyy-MM-dd"));
          form.setValue("end", format(departureDate as Date, "yyyy-MM-dd"));
          form.setValue("adult", Number(adult));
          form.setValue("child", Number(child));
        }
      });
    }
  };

  const resetForm = () => {
    setDateRange({
      startDate: new Date(),
      endDate: new Date(),
      key: "selection",
    });

    setGuest({
      adult: 1,
      child: 0,
    });

    form.reset();
  };

  const cancelForm = () => {
    setOpenGuestForm(false);
    setResult(null);
    setFormData(null);
    resetForm();
  };

  return (
    <div className="grid lg:grid-cols-7 gap-5 mb-6 h-full">
      <div className="col-span-1 lg:col-span-2 h-full">
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)}>
            <Card className="rounded">
              <CardHeader>
                <div className="flex items-center justify-between">
                  <CardTitle className="text-second">Availability</CardTitle>
                  <Link
                    href="/"
                    className="text-base text-main flex items-center"
                  >
                    <div className="flex items-center gap-2">
                      <IoArrowBackCircleOutline />
                      Go Back
                    </div>
                  </Link>
                </div>
              </CardHeader>
              <CardContent>
                <FormField
                  control={form.control}
                  name="start"
                  render={({field: {value, onChange, ...fieldProps}}) => (
                    <FormItem>
                      <FormControl>
                        <Input
                          type="hidden"
                          placeholder="shadcn"
                          {...fieldProps}
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <div className="space-y-3">
                  <Popover open={openCalendar} onOpenChange={setOpenCalendar}>
                    <PopoverTrigger
                      disabled={isPending || openGuestForm || isLoading}
                      className="w-full border px-5 py-2"
                    >
                      <div className="flex items-center">
                        <IoCalendarOutline className="mr-3" />
                        <span>
                          {dateRange.startDate?.toDateString() ===
                          dateRange.endDate?.toDateString()
                            ? "Check in - Check out"
                            : `${format(
                                dateRange.startDate as Date,
                                "MMM dd ",
                                {
                                  timeZone,
                                }
                              )} - ${format(
                                dateRange.endDate as Date,
                                "MMM dd, yyyy",
                                {timeZone}
                              )}`}
                        </span>
                      </div>
                    </PopoverTrigger>
                    <PopoverContent className="w-full z-0" align="center">
                      <div className="flex justify-end">
                        <PopoverClose className="bg-main w-fit rounded">
                          <X className="w-5 h-5 rounded-[50px] text-white" />
                        </PopoverClose>
                      </div>
                      <CalendarWidget
                        value={dateRange}
                        onChange={handleDateChange}
                        disabledDates={icalVilla?.data}
                      />
                    </PopoverContent>
                  </Popover>

                  <Popover>
                    <PopoverTrigger
                      className="w-full border px-5 py-2"
                      disabled={isPending || openGuestForm || isLoading}
                    >
                      <div className="flex items-center w-full">
                        <span className="mr-3">
                          <FaRegUser />
                        </span>
                        <span className="text-center">
                          {guest.adult + guest.child > 1
                            ? guest.adult + guest.child + " Guests"
                            : guest.adult + guest.child + " Guest"}
                        </span>
                      </div>
                    </PopoverTrigger>
                    <PopoverContent
                      className="PopoverContent z-0"
                      align="center"
                    >
                      <div className="flex justify-end">
                        <PopoverClose className="bg-main w-fit rounded">
                          <X className="w-5 h-5 rounded-[50px] text-white" />
                        </PopoverClose>
                      </div>
                      <div className="space-y-5 pt-5">
                        <Counter
                          title="Adult"
                          subtitle="Age 13 or Above"
                          value={guest.adult}
                          onChange={handleCounter}
                          action="adult"
                          disableLimit={1}
                          maxGuestLimit={8}
                          totalGuestCount={totalGuests}
                          className="!mb-0"
                          className2="bg-main text-white"
                          className3="items-center justify-between"
                        />
                        <Counter
                          title="Children"
                          subtitle="Age 2 to 12"
                          value={guest.child}
                          onChange={handleCounter}
                          action="child"
                          disableLimit={0}
                          maxGuestLimit={8}
                          totalGuestCount={totalGuests}
                          className="!mb-0"
                          className2="bg-main text-white"
                          className3="items-center justify-between"
                        />
                      </div>
                    </PopoverContent>
                  </Popover>
                </div>
              </CardContent>
              <CardFooter>
                <Button
                  type="submit"
                  className="w-full bg-main hover:bg-main/80 text-white flex items-center justify-center gap-2 disabled:cursor-not-allowed  "
                  disabled={isPending || openGuestForm || isLoading}
                >
                  {isPending ? <IoLockClosed /> : ""}
                  {openGuestForm ? <IoLockClosed /> : ""}
                  Check Availability
                </Button>
              </CardFooter>
            </Card>
          </form>
        </Form>
      </div>
      <div className="col-span-1 lg:col-span-5 h-full">
        <ResultForm
          result={result}
          loadingCheck={isPending}
          openForm={() => setOpenGuestForm(true)}
          cancelForm={cancelForm}
        />
      </div>
      {openGuestForm && <FormGuest data={formData} />}
    </div>
  );
}
